<?php
/**
 * WooCommerce - Function get Query
 */
function nasa_woocommerce_query($type, $post_per_page = -1, $cat = '', $paged = '', $not = array(), $deal_time = null) {
    global $woocommerce;
    if (!$woocommerce) {
        return array();
    }
    
    $page = $paged == '' ? ($paged = get_query_var('paged') ? (int) get_query_var('paged') : 1) : (int) $paged;

    $data = new WP_Query(nasa_woocommerce_query_args($type, $post_per_page, $cat, $page, $not, $deal_time));
    remove_filter('posts_clauses', 'nasa_order_by_rating_post_clauses');
    remove_filter('posts_clauses', 'nasa_order_by_recent_review_post_clauses');
    
    return $data;
}

/**
 * Order by rating review
 * 
 * @global type $wpdb
 * @param type $args
 * @return array
 */
function nasa_order_by_rating_post_clauses($args) {
    global $wpdb;

    $args['fields'] .= ', AVG(' . $wpdb->commentmeta . '.meta_value) as average_rating';
    $args['where'] .= ' AND (' . $wpdb->commentmeta . '.meta_key = "rating" OR ' . $wpdb->commentmeta . '.meta_key IS null) AND ' . $wpdb->comments . '.comment_approved=1 ';
    $args['join'] .= ' LEFT OUTER JOIN ' . $wpdb->comments . ' ON(' . $wpdb->posts . '.ID = ' . $wpdb->comments . '.comment_post_ID) LEFT JOIN ' . $wpdb->commentmeta . ' ON(' . $wpdb->comments . '.comment_ID = ' . $wpdb->commentmeta . '.comment_id) ';
    $args['orderby'] = 'average_rating DESC, ' . $wpdb->posts . '.post_date DESC';
    $args['groupby'] = $wpdb->posts . '.ID';

    return $args;
}

/**
 * Order by recent review
 * 
 * @global type $wpdb
 * @param type $args
 * @return array
 */
function nasa_order_by_recent_review_post_clauses($args) {
    global $wpdb;

    $args['where'] .= ' AND ' . $wpdb->comments . '.comment_approved=1 ';
    $args['join'] .= ' LEFT JOIN ' . $wpdb->comments . ' ON(' . $wpdb->posts . '.ID = ' . $wpdb->comments . '.comment_post_ID)';
    $args['orderby'] = $wpdb->comments . '.comment_date DESC, ' . $wpdb->comments . '.comment_post_ID DESC';
    $args['groupby'] = $wpdb->posts . '.ID';

    return $args;
}

/**
 * Build query for Nasa WooCommerce Products
 * 
 * @global type $woocommerce
 * @param type $type
 * @param type $post_per_page
 * @param type $cat
 * @param type $paged
 * @param type $not
 * @param type $deal_time
 * @return type
 */
function nasa_woocommerce_query_args($type, $post_per_page = -1, $cat = '', $paged = 1, $not = array(), $deal_time = null) {
    global $woocommerce;
    if (!$woocommerce) {
        return array();
    }

    $args = array(
        'post_type' => 'product',
        'posts_per_page' => $post_per_page,
        'post_status' => 'publish',
        'paged' => $paged
    );

    $args['meta_query'] = array();
    $args['meta_query'][] = $woocommerce->query->stock_status_meta_query();
    $args['tax_query'] = array('relation' => 'AND');
    
    switch ($type) {
        case 'best_selling':
            $args['ignore_sticky_posts'] = 1;
            
            $args['meta_key']   = 'total_sales';
            $args['order']      = 'DESC';
            $args['orderby']    = 'meta_value_num';
            
            $args['meta_query'][] = $woocommerce->query->visibility_meta_query();
            
            break;
        
        case 'featured_product':
            $args['ignore_sticky_posts'] = 1;

            $args['tax_query'][] = array(
                'taxonomy' => 'product_visibility',
                'field' => 'name',
                'terms' => 'featured'
            );
            
            $args['meta_query'][] = $woocommerce->query->visibility_meta_query();
            
            break;
        
        case 'top_rate':
            add_filter('posts_clauses', 'nasa_order_by_rating_post_clauses');
            $args['meta_query'][] = $woocommerce->query->visibility_meta_query();
            break;
        
        case 'recent_review':
            // nasa_order_by_recent_review_post_clauses
            add_filter('posts_clauses', 'nasa_order_by_recent_review_post_clauses');
            $args['meta_query'][] = $woocommerce->query->visibility_meta_query();

            break;
        
        case 'on_sale':
            $args['meta_query'][] = $woocommerce->query->visibility_meta_query();
            $args['post__in'] = array_merge(array(0), wc_get_product_ids_on_sale());
            
            break;
        
        case 'deals':
            $args['meta_query'][] = $woocommerce->query->visibility_meta_query();
            
            $args['meta_query'][] = array(
                'key' => '_sale_price_dates_from',
                'value' => NASA_TIME_NOW,
                'compare' => '<=',
                'type' => 'numeric'
            );
            
            $args['meta_query'][] = array(
                'key' => '_sale_price_dates_to',
                'value' => NASA_TIME_NOW,
                'compare' => '>',
                'type' => 'numeric'
            );
            
            $args['post_type'] = array('product', 'product_variation');

            if ($deal_time > 0) {
                $args['meta_query'][] = array(
                    'key' => '_sale_price_dates_to',
                    'value' => $deal_time,
                    'compare' => '>=',
                    'type' => 'numeric'
                );
            }
            
            $args['post__in'] = array_merge(array(0), nasa_get_product_deal_ids($cat));
            
            $args['orderby'] = 'date ID';
            $args['order']   = 'DESC';

            break;

        case 'recent_product':
        default:
            $args['orderby'] = 'date ID';
            $args['order']   = 'DESC';
            
            break;
    }

    if (!empty($not)) {
        $args['post__not_in'] = $not;
        if (!empty($args['post__in'])) {
            $args['post__in'] = array_diff($args['post__in'], $args['post__not_in']);
        }
    }

    if ($type !== 'deals' && $cat) {
        
        // Find by cat id
        if (is_numeric($cat)) {
            $args['tax_query'][] = array(
                'taxonomy' => 'product_cat',
                'field' => 'id',
                'terms' => array($cat)
            );
        }

        // Find by cat array id
        elseif (is_array($cat)) {
            $args['tax_query'][] = array(
                'taxonomy' => 'product_cat',
                'field' => 'id',
                'terms' => $cat
            );
        }

        // Find by slug
        elseif (is_string($cat)) {
            $args['tax_query'][] = array(
                'taxonomy' => 'product_cat',
                'field' => 'slug',
                'terms' => $cat
            );
        }
    }

    $product_visibility_terms = wc_get_product_visibility_term_ids();
    $arr_not_in = array($product_visibility_terms['exclude-from-catalog']);

    // Hide out of stock products.
    if ('yes' === get_option('woocommerce_hide_out_of_stock_items')) {
        $arr_not_in[] = $product_visibility_terms['outofstock'];
    }

    if (!empty($arr_not_in)) {
        $args['tax_query'][] = array(
            'taxonomy' => 'product_visibility',
            'field' => 'term_taxonomy_id',
            'terms' => $arr_not_in,
            'operator' => 'NOT IN',
        );
    }
    
    if (empty($args['orderby']) || empty($args['order'])) {
        $ordering_args      = WC()->query->get_catalog_ordering_args();
        $args['orderby']    = empty($args['orderby']) ? $ordering_args['orderby'] : $args['orderby'];
        $args['order']      = empty($args['order']) ? $ordering_args['order'] : $args['order'];
    }

    return apply_filters('nasa_woocommerce_query_args', $args);
}

/**
 * Get ids include for deal product
 * 
 * @global type $wpdb
 * @param type $cat
 * @return type
 */
function nasa_get_product_deal_ids ($cat = null) {
    $key = 'nasa_products_deal';
    if ($cat) {
        if (is_numeric($cat)) {
            $key .= '_cat_' . $cat;
        }
        
        if (is_array($cat)) {
            $key .= '_cats_' . implode('_', $cat);
        }
        
        if (is_string($cat)) {
            $key .= '_catslug_' . $cat;
        }
    }
    
    $product_ids = get_transient($key);
    
    if (!$product_ids) {
        $args = array(
            'post_type'         => 'product',
            'numberposts'       => -1,
            'post_status'       => 'publish',
            'fields'            => 'ids'
        );

        $args['tax_query'] = array('relation' => 'AND');

        $args['post__in'] = array_merge(array(0), wc_get_product_ids_on_sale());

        // Find by cat id
        if (is_numeric($cat) && $cat) {
            $args['tax_query'][] = array(
                'taxonomy' => 'product_cat',
                'field' => 'id',
                'terms' => array($cat)
            );
        }

        // Find by cat array id
        elseif (is_array($cat) && $cat) {
            $args['tax_query'][] = array(
                'taxonomy' => 'product_cat',
                'field' => 'id',
                'terms' => $cat
            );
        }

        // Find by slug
        elseif (is_string($cat) && $cat) {
            $args['tax_query'][] = array(
                'taxonomy' => 'product_cat',
                'field' => 'slug',
                'terms' => $cat
            );
        }

        $product_ids = get_posts($args);
        $product_ids_str = $product_ids ? implode(', ', $product_ids) : false;

        if ($product_ids_str) {
            global $wpdb;
            $variation_obj = $wpdb->get_results('SELECT ID FROM ' . $wpdb->posts . ' WHERE post_parent IN (' . $product_ids_str . ')');

            $variation_ids = $variation_obj ? wp_list_pluck($variation_obj, 'ID') : null;

            if ($variation_ids) {
                $product_ids = array_merge($product_ids, $variation_ids);
            }
        }

        set_transient($key, $product_ids, DAY_IN_SECONDS);
    }
    
    return $product_ids;
}

/**
 * Get Products by array id
 * 
 * @global type $woocommerce
 * @param type $ids
 * @return \WP_Query
 */
function nasa_get_products_by_ids($ids = array()) {
    global $woocommerce;
    if (!$woocommerce || empty($ids)) {
        return null;
    }
    
    $args = array(
        'post_type' => 'product',
        'post__in' => $ids,
        'posts_per_page' => count($ids),
        'post_status' => 'publish',
        'paged' => 1
    );
    
    return new WP_Query($args);
}

/**
 * Recommend product
 * @param type $cat
 */
add_action('nasa_recommend_product', 'nasa_get_recommend_product', 10, 1);
function nasa_get_recommend_product($cat = null) {
    global $nasa_opt, $woocommerce;

    if (!$woocommerce || (isset($nasa_opt['enable_recommend_product']) && $nasa_opt['enable_recommend_product'] != '1')) {
        return '';
    }

    $columns_number = isset($nasa_opt['recommend_columns_desk']) ? (int) $nasa_opt['recommend_columns_desk'] : 5;

    $columns_number_small = isset($nasa_opt['recommend_columns_small']) ? $nasa_opt['recommend_columns_small'] : 2;
    $columns_number_small_slider = $columns_number_small == '1.5-cols' ? '1.5' : (int) $columns_number_small;
    
    $columns_number_tablet = isset($nasa_opt['recommend_columns_tablet']) ? (int) $nasa_opt['recommend_columns_tablet'] : 3;

    $number = (isset($nasa_opt['recommend_product_limit']) && ((int) $nasa_opt['recommend_product_limit'] >= $columns_number)) ? (int) $nasa_opt['recommend_product_limit'] : 9;

    $loop = nasa_woocommerce_query('featured_product', $number, (int) $cat ? (int) $cat : null, 1);
    if ($loop->found_posts) {
        ?>
        <div class="row margin-bottom-50 nasa-recommend-product">
            <div class="large-12 columns">
                <div class="woocommerce">
                    <?php
                    $type = null;
                    $height_auto = 'false';
                    $arrows = 1;
                    $title_shortcode = esc_html__('Recommend Products', 'nasa-core');
                    
                    $nasa_args = array(
                        'loop' => $loop,
                        'cat' => $cat,
                        'columns_number' => $columns_number,
                        'columns_number_small_slider' => $columns_number_small_slider,
                        'columns_number_tablet' => $columns_number_tablet,
                        'number' => $number,
                        'type' => $type,
                        'height_auto' => $height_auto,
                        'arrows' => $arrows,
                        'title_shortcode' => $title_shortcode,
                        'title_align' => 'center',
                        'nasa_opt' => $nasa_opt
                    );
                    
                    nasa_template('products/nasa_products/carousel.php', $nasa_args);
                    ?>
                </div>
                <?php
                if (isset($nasa_opt['recommend_product_position']) && $nasa_opt['recommend_product_position'] == 'top') :
                    echo '<hr class="nasa-separator" />';
                endif;
                ?>
            </div>
        </div>
        <?php
    }
}

/**
 * Get product Deal by id
 * 
 * @param type $id
 * @return type
 */
function nasa_getProductDeals($id = null) {
    if (!(int) $id || !function_exists('wc_get_product')) {
        return null;
    }

    $product = wc_get_product((int) $id);
    if ($product) {
        $time_sale = $time_from = false;
        $product_type = $product->get_type();

        if ($product_type == 'variable') {
            $args = array(
                'fields' => 'ids',
                'post_type' => 'product_variation',
                'post_parent' => (int) $id,
                'posts_per_page' => 100,
                'post_status' => 'publish',
                'orderby' => 'ID',
                'order' => 'ASC',
                'paged' => 1
            );

            $children = new WP_Query($args);
            if (!empty($children->posts)) {
                foreach ($children->posts as $variable) {
                    $time_sale = get_post_meta($variable, '_sale_price_dates_to', true);
                    $time_from = get_post_meta($variable, '_sale_price_dates_from', true);
                    if ($time_sale && $time_sale > NASA_TIME_NOW) {
                        break;
                    }
                }
            }
        }

        if (!$time_sale && $product_type == 'simple') {
            $time_sale = !$time_sale ? get_post_meta((int) $id, '_sale_price_dates_to', true) : $time_sale;
            $time_from = get_post_meta((int) $id, '_sale_price_dates_from', true);
        }

        if ($time_sale > NASA_TIME_NOW && $time_from && $time_from < NASA_TIME_NOW) {
            $product->time_sale = $time_sale;
            
            return $product;
        }
    }

    return null;
}

/**
 * Get products in grid
 * 
 * @param type $notid
 * @param type $catIds
 * @param type $type
 * @param type $limit
 * @return type
 */
function nasa_getProductGrid($notid = null, $catIds = null, $type = 'best_selling', $limit = 6) {
    $notIn = $notid ? array($notid) : array();
    
    return nasa_woocommerce_query($type, $limit, $catIds, 1, $notIn);
}

/**
 * Set cookie products viewed
 */
remove_action('template_redirect', 'wc_track_product_view', 25);
add_action('template_redirect', 'nasa_set_products_viewed', 20);
function nasa_set_products_viewed() {
    global $nasa_opt;

    if (!class_exists('WooCommerce') || !is_singular('product') || (isset($nasa_opt['enable-viewed']) && !$nasa_opt['enable-viewed'])) {
        return;
    }

    global $post;

    $product_id = isset($post->ID) ? (int) $post->ID : 0;
    if ($product_id) {
        $limit = !isset($nasa_opt['limit_product_viewed']) || !(int) $nasa_opt['limit_product_viewed'] ?
            12 : (int) $nasa_opt['limit_product_viewed'];

        $list_viewed = !empty($_COOKIE[NASA_COOKIE_VIEWED]) ? explode('|', $_COOKIE[NASA_COOKIE_VIEWED]) : array();
        if (!in_array((int) $product_id, $list_viewed)) {
            $new_array = array(0 => $product_id);
            
            for ($i = 1; $i < $limit; $i++) {
                if (isset($list_viewed[$i-1])) {
                    $new_array[$i] = $list_viewed[$i-1];
                }
            }
            
            $new_array_str = !empty($new_array) ? implode('|', $new_array) : '';
            setcookie(NASA_COOKIE_VIEWED, $new_array_str, 0, COOKIEPATH, COOKIE_DOMAIN, false, false);
        }
    }
}

/**
 * Get cookie products viewed
 */
function nasa_get_products_viewed() {
    global $nasa_opt;
    $query = null;

    if (!class_exists('WooCommerce') || (isset($nasa_opt['enable-viewed']) && !$nasa_opt['enable-viewed'])) {
        return $query;
    }

    $viewed_products = !empty($_COOKIE[NASA_COOKIE_VIEWED]) ? explode('|', $_COOKIE[NASA_COOKIE_VIEWED]) : array();
    if (!empty($viewed_products)) {

        $limit = !isset($nasa_opt['limit_product_viewed']) || !(int) $nasa_opt['limit_product_viewed'] ? 12 : (int) $nasa_opt['limit_product_viewed'];

        $query_args = array(
            'posts_per_page' => $limit,
            'no_found_rows' => 1,
            'post_status' => 'publish',
            'post_type' => 'product',
            'post__in' => $viewed_products,
            'orderby' => 'post__in',
        );

        if ('yes' === get_option('woocommerce_hide_out_of_stock_items')) {
            $query_args['tax_query'] = array(
                array(
                    'taxonomy' => 'product_visibility',
                    'field' => 'name',
                    'terms' => 'outofstock',
                    'operator' => 'NOT IN',
                ),
            );
        }

        $query = new WP_Query($query_args);
    }

    return $query;
}

/**
 * Static Viewed Sidebar
 */
add_action('nasa_static_content', 'nasa_static_viewed_sidebar', 15);
function nasa_static_viewed_sidebar() {
    global $nasa_opt;
    if (!class_exists('WooCommerce') || (isset($nasa_opt['enable-viewed']) && !$nasa_opt['enable-viewed'])) {
        return;
    }
    
    $nasa_viewed_style = isset($nasa_opt['style-viewed']) ? esc_attr($nasa_opt['style-viewed']) : 'style-1'; ?>
    <!-- viewed product -->
    <div id="nasa-viewed-sidebar" class="nasa-static-sidebar <?php echo esc_attr($nasa_viewed_style); ?>">
        <div class="viewed-close nasa-sidebar-close">
            <h3 class="nasa-tit-viewed nasa-sidebar-tit text-center">
                <?php echo esc_html__("Recently Viewed", 'nasa-core'); ?>
            </h3>
            <a href="javascript:void(0);" title="<?php esc_attr_e('Close', 'nasa-core'); ?>"><?php esc_html_e('Close', 'nasa-core'); ?></a>
        </div>
        
        <div id="nasa-viewed-sidebar-content" class="nasa-absolute">
            <div class="nasa-loader"></div>
        </div>
    </div>
    <?php
}

/**
 * Viewed icon button
 */
add_action('nasa_static_group_btns', 'nasa_static_viewed_btns', 15);
function nasa_static_viewed_btns() {
    global $nasa_opt;
    if (!class_exists('WooCommerce') || (isset($nasa_opt['enable-viewed']) && !$nasa_opt['enable-viewed'])) {
        return;
    } ?>
    
    <?php
    $nasa_viewed_icon = 'nasa-tip nasa-tip-left ';
    $nasa_viewed_icon .= isset($nasa_opt['style-viewed-icon']) ? esc_attr($nasa_opt['style-viewed-icon']) : 'style-1';
    ?>
    <a id="nasa-init-viewed" class="<?php echo esc_attr($nasa_viewed_icon); ?>" href="javascript:void(0);" data-tip="<?php esc_attr_e('Recently Viewed', 'nasa-core'); ?>" title="<?php esc_attr_e('Recently Viewed', 'nasa-core'); ?>">
        <i class="pe-icon pe-7s-clock"></i>
    </a>
    <?php
}

/**
 * Get product meta value
 */
function nasa_get_product_meta_value($post_id, $field_id = null) {
    global $nasa_product_meta;
    
    if (isset($nasa_product_meta[$post_id])) {
        $meta_value = $nasa_product_meta[$post_id];
    } else {
        $meta_value = get_post_meta($post_id, 'wc_productdata_options', true);
        $nasa_product_meta = !$nasa_product_meta ? array() : $nasa_product_meta;
        $nasa_product_meta[$post_id] = $meta_value;
        $GLOBALS['nasa_product_meta'] = $nasa_product_meta;
    }
    
    if (isset($meta_value[0]) && $field_id) {
        return isset($meta_value[0][$field_id]) ? $meta_value[0][$field_id] : '';
    }

    return isset($meta_value[0]) ? $meta_value[0] : $meta_value;
}

/**
 * variation gallery images
 */
add_filter('woocommerce_available_variation', 'nasa_variation_gallery_images');
function nasa_variation_gallery_images($variation) {
    global $nasa_opt;
    if (!isset($nasa_opt['gallery_images_variation']) || $nasa_opt['gallery_images_variation']) {
        if (!isset($variation['nasa_gallery_variation'])) {
            $variation['nasa_gallery_variation'] = array();
            $variation['nasa_variation_back_img'] = '';
            $gallery = get_post_meta($variation['variation_id'], 'nasa_variation_gallery_images', true);

            if ($gallery) {
                $variation['nasa_gallery_variation'] = $gallery;
                $galleryIds = explode(',', $gallery);
                $back_id = isset($galleryIds[0]) && (int) $galleryIds[0] ? (int) $galleryIds[0] : false;
                $image_size = apply_filters('single_product_archive_thumbnail_size', 'shop_catalog');
                $image_back = $back_id ? wp_get_attachment_image_src($back_id, $image_size) : null;
                $variation['nasa_variation_back_img'] = isset($image_back[0]) ? $image_back[0] : '';
            }
        }
    }
    
    return $variation;
}

/**
 * Enable Gallery images variation in front-end
 */
add_action('woocommerce_after_add_to_cart_button', 'nasa_enable_variation_gallery_images', 30);
function nasa_enable_variation_gallery_images() {
    global $product, $nasa_opt;
    
    if (!isset($nasa_opt['gallery_images_variation']) || $nasa_opt['gallery_images_variation']) {
        $productType = $product->get_type();
        if ($productType == 'variable' || $productType == 'variation') {
            $mainProduct = ($productType == 'variation') ?
                wc_get_product(wp_get_post_parent_id($product->get_id())) : $product;
            
            if (!$mainProduct) {
                return;
            }

            $variations = $mainProduct->get_available_variations();
            foreach ($variations as $vari) {
                if (isset($vari['nasa_gallery_variation']) && !empty($vari['nasa_gallery_variation'])) {
                    echo '<input type="hidden" name="nasa-gallery-variation-supported" class="nasa-gallery-variation-supported" value="1" />';
                    
                    return;
                }
            }
        }
    }
}

/**
 * Size Guide Product - Delivery & Return
 */
add_action('woocommerce_single_product_summary', 'nasa_single_product_popup_nodes', 35);
function nasa_single_product_popup_nodes() {
    global $nasa_opt, $product;
    
    /**
     * Size Guide - New Feature get content from static Block
     */
    $single_product = false;
    $size_guide = false;
    
    /**
     * Get size_guide from category
     */
    $term_id = nasa_root_term_id();

    if ($term_id) {
        $size_guide_cat = get_term_meta($term_id, 'cat_size_guide_block', true);
        
        if ($size_guide_cat && $size_guide_cat != '-1') {
            $size_guide = nasa_get_block($size_guide_cat);
        }
        
        if ($size_guide_cat == '-1') {
            $size_guide = 'not-show';
        }
    }

    /**
     * Get size_guide from Theme Options
     */
    if (!$size_guide && isset($nasa_opt['size_guide_product']) && $nasa_opt['size_guide_product']) {
        $size_guide = nasa_get_block($nasa_opt['size_guide_product']);
    }
    
    /**
     * Not show from Category
     */
    if ($size_guide == 'not-show') {
        $size_guide = false;
    }
    
    /**
     * Delivery & Return
     */
    $delivery_return = false;
    if (isset($nasa_opt['delivery_return_single_product']) && $nasa_opt['delivery_return_single_product']) {
        $delivery_return = nasa_get_block($nasa_opt['delivery_return_single_product']);
    }
    
    /**
     * Ask a Question
     */
    $ask_a_question = false;
    if (isset($nasa_opt['ask_a_question']) && $nasa_opt['ask_a_question']) {
        $ask_a_question = shortcode_exists('contact-form-7') ? do_shortcode('[contact-form-7 id="' . ((int) $nasa_opt['ask_a_question']) . '"]') : false;
        
        if ($ask_a_question == '[contact-form-7 404 "Not Found"]') {
            $ask_a_question = false;
        }
        
        if ($ask_a_question) {
            global $product;
            $single_product = $product;
        }
    }
    
    /**
     * Request a Call Back
     */
    $request_a_callback = false;
    if (isset($nasa_opt['request_a_callback']) && $nasa_opt['request_a_callback']) {
        $request_a_callback = shortcode_exists('contact-form-7') ? do_shortcode('[contact-form-7 id="' . ((int) $nasa_opt['request_a_callback']) . '"]') : false;
        
        if ($request_a_callback == '[contact-form-7 404 "Not Found"]') {
            $request_a_callback = false;
        }
        
        if (!$single_product && $request_a_callback) {
            global $product;
            $single_product = $product;
        }
    }
    
    /**
     * Args Template
     */
    $nasa_args = array(
        'size_guide' => $size_guide,
        'delivery_return' => $delivery_return,
        'ask_a_question' => $ask_a_question,
        'request_a_callback' => $request_a_callback,
        'single_product' => $single_product
    );
    
    /**
     * Include template
     */
    nasa_template('products/nasa_single_product/nasa-single-product-popup-nodes.php', $nasa_args);
}

/**
 * Viewed icon button
 */
add_action('nasa_static_group_btns', 'nasa_static_request_callbacl', 12);
function nasa_static_request_callbacl() {
    global $nasa_opt;
    
    if (!isset($nasa_opt['request_a_callback']) || !$nasa_opt['request_a_callback']) {
        return;
    }
    
    if (!class_exists('WooCommerce') || !is_product()) {
        return;
    } ?>
    <a class="nasa-node-popup hidden-tag nasa-tip nasa-tip-left" href="javascript:void(0);" data-target="#nasa-content-request-a-callback" data-tip="<?php echo esc_attr__('Request a Call Back', 'nasa-core'); ?>" title="<?php echo esc_attr__('Request a Call Back', 'nasa-core'); ?>">
        <i class="nasa-icon icon-nasa-headphone"></i>
    </a>
    <?php
}

/**
 * After Add To Cart Button
 */
add_action('woocommerce_after_add_to_cart_form', 'nasa_after_add_to_cart_form');
function nasa_after_add_to_cart_form() {
    global $nasa_opt;
    
    if (isset($nasa_opt['after_single_addtocart_form']) && $nasa_opt['after_single_addtocart_form']) {
        echo nasa_get_block($nasa_opt['after_single_addtocart_form']);
    }
}

/**
 * After Process To Checkout Button
 */
add_action('woocommerce_proceed_to_checkout', 'nasa_after_process_checkout_button', 100);
function nasa_after_process_checkout_button() {
    global $nasa_opt;
    
    if (isset($nasa_opt['after_process_checkout']) && $nasa_opt['after_process_checkout']) {
        echo nasa_get_block($nasa_opt['after_process_checkout']);
    }
}

/**
 * After Cart Table
 */
add_action('woocommerce_after_cart_table', 'nasa_after_cart_table');
function nasa_after_cart_table() {
    global $nasa_opt;
    
    if (isset($nasa_opt['after_cart_table']) && $nasa_opt['after_cart_table']) {
        echo nasa_get_block($nasa_opt['after_cart_table']);
    }
}

/**
 * After Cart content
 */
add_action('woocommerce_after_cart', 'nasa_after_cart', 5);
function nasa_after_cart() {
    global $nasa_opt;
    
    if (isset($nasa_opt['after_cart']) && $nasa_opt['after_cart']) {
        echo nasa_get_block($nasa_opt['after_cart']);
    }
}

/**
 * After Place Order Button
 */
add_action('woocommerce_review_order_after_payment', 'nasa_after_place_order_button');
function nasa_after_place_order_button() {
    global $nasa_opt;
    
    if (isset($nasa_opt['after_place_order']) && $nasa_opt['after_place_order']) {
        echo nasa_get_block($nasa_opt['after_place_order']);
    }
}

/**
 * After review order
 */
if (defined('NASA_THEME_ACTIVE') && NASA_THEME_ACTIVE) {
    add_action('nasa_checkout_after_order_review', 'nasa_after_review_order_payment');
} else {
    add_action('woocommerce_checkout_after_order_review', 'nasa_after_review_order_payment');
}
function nasa_after_review_order_payment() {
    global $nasa_opt;
    
    if (isset($nasa_opt['after_review_order']) && $nasa_opt['after_review_order']) {
        echo nasa_get_block($nasa_opt['after_review_order']);
    }
}

/**
 * Custom Slug Nasa Custom Categories
 */
add_filter('nasa_taxonomy_custom_cateogory', 'nasa_custom_slug_categories');
function nasa_custom_slug_categories($slug) {
    global $nasa_opt;
    
    if (!class_exists('WooCommerce') || !isset($nasa_opt['enable_nasa_custom_categories']) || !$nasa_opt['enable_nasa_custom_categories'] || !isset($nasa_opt['nasa_custom_categories_slug']) || trim($nasa_opt['nasa_custom_categories_slug']) === '') {
        return $slug;
    }
    
    $new_slug = sanitize_title(trim($nasa_opt['nasa_custom_categories_slug']));
    
    return $new_slug;
}

/**
 * Custom nasa-taxonomy
 */
add_action('nasa_before_archive_products', 'nasa_custom_filter_taxonomies');
function nasa_custom_filter_taxonomies() {
    global $nasa_opt, $wp_query, $nasa_root_term_id;
    
    if (!class_exists('WooCommerce') || !isset($nasa_opt['enable_nasa_custom_categories']) || !$nasa_opt['enable_nasa_custom_categories']) {
        return;
    }
    
    if (!isset($nasa_root_term_id)) {
        $is_product_cat = is_product_category();
        $current_cat = null;

        $rootCatId = 0;

        if ($is_product_cat) {
            $current_cat = $wp_query->get_queried_object();
        }

        if ($current_cat && isset($current_cat->term_id)) {
            if (isset($current_cat->parent) && $current_cat->parent == 0) {
                $rootCatId = $current_cat->term_id;
            } else {
                $ancestors = get_ancestors($current_cat->term_id, 'product_cat');
                $rootCatId = end($ancestors);
            }
        }

        $GLOBALS['nasa_root_term_id'] = $rootCatId ? $rootCatId : 0;
        $nasa_root_term_id = $rootCatId;
    }
    
    $show = '';
    if ($nasa_root_term_id) {
        $show = get_term_meta($nasa_root_term_id, 'nasa_custom_tax', true);
    }
    
    if ($show == '') {
        $show = isset($nasa_opt['archive_product_nasa_custom_categories']) && $nasa_opt['archive_product_nasa_custom_categories'] ? 'show' : 'hide';
    }
    
    if ($show === 'hide') {
        return;
    }

    $class = 'large-12 columns';
    $max = isset($nasa_opt['max_level_nasa_custom_categories']) ? (int) $nasa_opt['max_level_nasa_custom_categories'] : 3;
    $max_level = $max > 3 || $max < 1 ? 3 : $max;
    echo '<div class="' . esc_attr($class) . '">';
    echo do_shortcode('[nasa_product_nasa_categories deep_level="' . esc_attr($max_level) . '" el_class="margin-top-15 mobile-margin-top-10"]');
    echo '</div>';
}

/**
 * 360 Degree Product Viewer
 */
add_action('nasa_single_buttons', 'nasa_360_product_viewer', 25);
function nasa_360_product_viewer() {
    global $nasa_opt, $product;
    if (isset($nasa_opt['product_360_degree']) && !$nasa_opt['product_360_degree']) {
        return;
    }
    
    /**
     * 360 Degree Product Viewer
     * 
     * jQuery lib
     */
    wp_enqueue_script('jquery-threesixty', NASA_CORE_PLUGIN_URL . 'assets/js/min/threesixty.min.js', array('jquery'), null, true);
    
    $idImgs = nasa_get_product_meta_value($product->get_id(), '_product_360_degree');
    $idImgs_str = $idImgs ? trim($idImgs, ',') : '';
    $idImgs_arr = $idImgs_str !== '' ? explode(',', $idImgs_str) : array();
    
    if (empty($idImgs_arr)) {
        return;
    }
    
    $img_src = array();
    $width = apply_filters('nasa_360_product_viewer_width_default', 500);
    $height = apply_filters('nasa_360_product_viewer_height_default', 500);
    $set = false;
    foreach ($idImgs_arr as $id) {
        $image_full = wp_get_attachment_image_src($id, 'full');
        if (isset($image_full[0])) {
            $img_src[] = $image_full[0];
            if (!$set) {
                $set = true;
                $width = isset($image_full[1]) ? $image_full[1] : $width;
                $height = isset($image_full[2]) ? $image_full[2] : $height;
                
            }
        } else {
            $img_src[] = wp_get_attachment_url($id);
        }
    }
    
    if (!empty($img_src)) {
        $img_src_json = wp_json_encode($img_src);
        $dataimgs = function_exists('wc_esc_json') ?
            wc_esc_json($img_src_json) : _wp_specialchars($img_src_json, ENT_QUOTES, 'UTF-8', true);
        
        echo '<a id="nasa-360-degree" class="nasa-360-degree-popup nasa-tip nasa-tip-right" href="javascript:void(0);" data-close="' . esc_attr__('Close', 'nasa-core') . '" data-imgs="' . $dataimgs . '" data-width="' . $width . '" data-height="' . $height . '" data-tip="' . esc_html__('360&#176; View', 'nasa-core') . '">' . esc_html__('360&#176;', 'nasa-core')  . '</a>';
    }
}

/**
 *
 * Add tab Bought Together
 */
add_filter('woocommerce_product_tabs', 'nasa_accessories_product_tab');
function nasa_accessories_product_tab($tabs) {
    global $product;
    
    if ($product && 'simple' === $product->get_type()) {
        $productIds = get_post_meta($product->get_id(), '_accessories_ids', true);
        if (!empty($productIds)) {
            $GLOBALS['accessories_ids'] = $productIds;
            $tabs['accessories_content'] = array(
                'title'     => esc_html__('Bought Together', 'nasa-core'),
                'priority'  => apply_filters('nasa_bought_together_tab_priority', 5),
                'callback'  => 'nasa_accessories_product_tab_content'
            );
        }
    }

    return $tabs;
}

/**
 * Content Bought Together of the current Product
 */
function nasa_accessories_product_tab_content() {
    global $product, $accessories_ids, $nasa_opt;
    if (!$product || !$accessories_ids) {
        return;
    }

    $accessories = array();
    foreach ($accessories_ids as $accessories_id) {
        $product_acc = wc_get_product($accessories_id);
        if (
            is_object($product_acc) &&
            $product_acc->get_status() === 'publish' &&
            $product_acc->get_type() == 'simple'
        ) {
            $accessories[] = $product_acc;
        }
    }

    if (empty($accessories)) {
        return;
    }
    
    $nasa_args = array(
        'nasa_opt' => $nasa_opt,
        'product' => $product,
        'accessories_ids' => $accessories_ids,
        'accessories' => $accessories,
    );

    nasa_template('products/nasa_single_product/nasa-single-product-accessories-tab-content.php', $nasa_args);
}

/**
 *
 * Add tab Technical Specifications
 */
add_filter('woocommerce_product_tabs', 'nasa_specifications_product_tab');
function nasa_specifications_product_tab($tabs) {
    global $nasa_specifications, $product;
    if (!$product) {
        return;
    }
    
    $product_id = $product->get_id();
    if (!isset($nasa_specifications[$product_id])) {
        $specifications = nasa_get_product_meta_value($product_id, 'nasa_specifications');
        $nasa_specifications[$product->get_id()] = $specifications;
        $GLOBALS['nasa_specifications'] = $nasa_specifications;
    }
    
    if ($nasa_specifications[$product_id] == '') {
        return $tabs;
    }
    
    $tabs['specifications'] = array(
        'title'     => esc_html__('Specifications', 'nasa-core'),
        'priority'  => apply_filters('nasa_specifications_tab_priority', 15),
        'callback'  => 'nasa_specifications_product_tab_content'
    );

    return $tabs;
}

/**
 * Content Technical Specifications of the current Product
 */
function nasa_specifications_product_tab_content() {
    global $product, $nasa_specifications;
    if (!$product || !isset($nasa_specifications[$product->get_id()])) {
        return;
    }

    echo do_shortcode($nasa_specifications[$product->get_id()]);
}

/**
 * Loop layout buttons
 */
add_action('wp_head', 'nasa_loop_layout_buttons');
function nasa_loop_layout_buttons() {
    if (!class_exists('WooCommerce')) {
        return false;
    }
    
    global $nasa_opt;
    $root_term_id = nasa_root_term_id();
    
    /**
     * Category products
     */
    if ($root_term_id) {
        $type_override = get_term_meta($root_term_id, 'nasa_loop_layout_buttons', true);
        if ($type_override) {
            $nasa_opt['loop_layout_buttons'] = $type_override;
        }
    }
    
    /**
     * Pages
     */
    else {
        global $wp_query;
        
        $page_id = false;
        $is_shop = is_shop();
        $is_product_taxonomy = is_product_taxonomy();

        /**
         * Shop
         */
        if ($is_shop || $is_product_taxonomy) {
            $pageShop = wc_get_page_id('shop');

            if ($pageShop > 0) {
                $page_id = $pageShop;
            }
        }

        /**
         * Page
         */
        else {
            $page_id = $wp_query->get_queried_object_id();
        }

        /**
         * Swith header structure
         */
        if ($page_id) {
            $type_override = get_post_meta($page_id, '_nasa_loop_layout_buttons', true);
            if (!empty($type_override)) {
                $nasa_opt['loop_layout_buttons'] = $type_override;
            }
        }
    }
    
    $GLOBALS['nasa_opt'] = $nasa_opt;
}

/**
 * Get Root Term id
 * 
 * @global type $nasa_root_term_id
 * @global type $product
 * @global type $wp_query
 * @global type $product
 * @return boolean
 */
function nasa_root_term_id() {
    if (!class_exists('WooCommerce')) {
        return false;
    }
    
    global $wp_query, $nasa_root_term_id;
    
    if (!isset($nasa_root_term_id)) {
        $is_product = is_product();
        $is_product_cat = is_product_category();
        $current_cat = null;
        
        /**
         * For Quick view
         */
        if (isset($_REQUEST['wc-ajax']) && $_REQUEST['wc-ajax'] === 'nasa_quick_view') {
            global $product;
            if (!$product) {
                return false;
            }

            $is_product = true;
        }

        $rootCatId = 0;
        if ($is_product) {
            global $post;

            $product_cats = get_the_terms($post->ID, 'product_cat');
            if ($product_cats) {
                foreach ($product_cats as $cat) {
                    $current_cat = $cat;
                    if ($cat->parent == 0) {
                        break;
                    }
                }
            }
        }

        elseif ($is_product_cat) {
            $current_cat = $wp_query->get_queried_object();
        }

        if ($current_cat && isset($current_cat->term_id)) {
            if (isset($current_cat->parent) && $current_cat->parent == 0) {
                $rootCatId = $current_cat->term_id;
            } else {
                $ancestors = get_ancestors($current_cat->term_id, 'product_cat');
                $rootCatId = end($ancestors);
            }
        }

        $GLOBALS['nasa_root_term_id'] = $rootCatId ? $rootCatId : 0;
        $nasa_root_term_id = $rootCatId;
    }
    
    return $nasa_root_term_id;
}
