<?php
/**
 * Nasa Cache Files
 */
class Nasa_Caching {
    
    public static $_live_time = 3600;
    
    protected static $_subkey = '_folder_';

    /**
     * Set Cache
     * 
     * @param type $key
     * @param type $content
     * @param type $folder
     */
    public static function set_content($key, $content, $folder = 'nasa-core') {
        global $nasa_opt;
        
        if (isset($nasa_opt['enable_nasa_cache']) && !$nasa_opt['enable_nasa_cache']) {
            return false;
        }
        
        /**
         * Cache Mode
         */
        $cache_mode = isset($nasa_opt['nasa_cache_mode']) && $nasa_opt['nasa_cache_mode'] == 'transient' ? 'transient' : 'file';
        
        /**
         * Set Transient
         */
        if ($cache_mode == 'transient') {
            return self::_set_content_transient($key, $content, $folder);
        }
        
        /**
         * Set File
         */
        return self::_set_content_file($key, $content, $folder);
    }
    
    /**
     * Set Transient data
     * 
     * @param type $key
     * @param type $content
     * @param type $folder
     */
    protected static function _set_content_transient($key, $content, $folder) {
        global $nasa_opt;
        
        if (isset($nasa_opt['nasa_cache_expire']) && (int) $nasa_opt['nasa_cache_expire']) {
            self::$_live_time = (int) $nasa_opt['nasa_cache_expire'];
        }
        
        $data = wp_json_encode(array('data' => $content));
        $key_name = 'nasa_cache_' . md5($key) . self::$_subkey . md5($folder);
        
        return set_transient($key_name, $data, self::$_live_time);
    }
    
    /**
     * Set File data
     * 
     * @param type $key
     * @param type $content
     * @param type $folder
     */
    protected static function _set_content_file($key, $content, $folder) {
        global $wp_filesystem, $nasa_cache_dir;

        if (!isset($nasa_cache_dir) || !$nasa_cache_dir) {
            $upload_dir = wp_upload_dir();
            $nasa_cache_dir = $upload_dir['basedir'] . '/nasa-cache';

            $GLOBALS['nasa_cache_dir'] = $nasa_cache_dir;
        }

        // Initialize the WP filesystem, no more using 'file-put-contents' function
        if (empty($wp_filesystem)) {
            require_once ABSPATH . '/wp-admin/includes/file.php';
            WP_Filesystem();
        }

        if (!defined('FS_CHMOD_FILE')) {
            define('FS_CHMOD_FILE', (fileperms(ABSPATH . 'index.php') & 0777 | 0644));
        }

        /**
         * Create new root cache
         */
        if (!$wp_filesystem->is_dir($nasa_cache_dir)) {
            if (!wp_mkdir_p($nasa_cache_dir)){
                return false;
            }
        }

        $folder_cache = $nasa_cache_dir . '/' . $folder;
        if (!$wp_filesystem->is_dir($folder_cache)) {   
            /**
             * Create folder cache products
             */
            if (!wp_mkdir_p($folder_cache)){
                return false;
            }
        }

        /**
         * Create htaccess file
         */
        $htaccess = $folder_cache . '/.htaccess';
        if (!is_file($htaccess)) {
            if (!$wp_filesystem->put_contents($htaccess, 'Deny from all', FS_CHMOD_FILE)) {
                return false;
            }
        }

        /**
         * Set cache file
         */
        $filename = $folder_cache . '/' . md5($key) . '.html';
        if (!$wp_filesystem->put_contents($filename, $content, FS_CHMOD_FILE)) {
            return false;
        }

        return true;
    }

    /**
     * Get Cache
     * 
     * @param type $key
     * @param type $folder
     */
    public static function get_content($key, $folder = 'nasa-core') {
        global $nasa_opt;
        
        if (isset($nasa_opt['enable_nasa_cache']) && !$nasa_opt['enable_nasa_cache']) {
            return false;
        }
        
        /**
         * Cache Mode
         */
        $cache_mode = isset($nasa_opt['nasa_cache_mode']) && $nasa_opt['nasa_cache_mode'] == 'transient' ? 'transient' : 'file';
        
        /**
         * Get Transient data
         */
        if ($cache_mode == 'transient') {
            return self::_get_content_transient($key, $folder);
        }
        
        /**
         * Get File data
         */
        return self::_get_content_file($key, $folder);
    }
    
    /**
     * Get Transient data
     * 
     * @param type $key
     * @param type $content
     * @param type $folder
     */
    protected static function _get_content_transient($key, $folder) {
        $key_name = 'nasa_cache_' . md5($key) . self::$_subkey . md5($folder);
        $transient = get_transient($key_name);
        if ($transient) {
            $jsonData = json_decode($transient);
            
            return ($jsonData && isset($jsonData->data)) ? $jsonData->data : null;
        }
        
        return false;
    }
    
    /**
     * Get File data
     * 
     * @param type $key
     * @param type $content
     * @param type $folder
     */
    protected static function _get_content_file($key, $folder) {
        global $nasa_opt, $wp_filesystem, $nasa_cache_dir;

        if (!isset($nasa_cache_dir) || !$nasa_cache_dir) {
            $upload_dir = wp_upload_dir();
            $nasa_cache_dir = $upload_dir['basedir'] . '/nasa-cache';

            $GLOBALS['nasa_cache_dir'] = $nasa_cache_dir;
        }

        // Initialize the WP filesystem, no more using 'file-put-contents' function
        if (empty($wp_filesystem)) {
            require_once ABSPATH . '/wp-admin/includes/file.php';
            WP_Filesystem();
        }

        $filename = $nasa_cache_dir . '/' . $folder . '/' . md5($key) . '.html';
        if (!is_file($filename)) {
            return false;
        }

        $time = filemtime($filename);
        if (isset($nasa_opt['nasa_cache_expire']) && (int) $nasa_opt['nasa_cache_expire']) {
            self::$_live_time = (int) $nasa_opt['nasa_cache_expire'];
        }
        if ($time + self::$_live_time < NASA_TIME_NOW) {
            return false;
        }

        return $wp_filesystem->get_contents($filename);
    }
    
    /**
     * Delete cache by key
     * 
     * @global string $nasa_cache_dir
     * @param type $key
     * @param type $folder
     * @return boolean
     */
    public static function delete_cache_by_key($key, $folder = 'nasa-core') {
        global $nasa_opt;
        
        /**
         * Cache Mode
         */
        $cache_mode = isset($nasa_opt['nasa_cache_mode']) && $nasa_opt['nasa_cache_mode'] == 'transient' ? 'transient' : 'file';
        
        try {
            /**
             * Set Transient
             */
            if ($cache_mode == 'transient') {
                self::_delete_transient_by_key($key, $folder);
            }

            /**
             * Set File
             */
            self::_delete_file_by_key($key, $folder);
            
            return true;
        } catch (Exception $exc) {
            return false;
        }
    }
    
    /**
     * Delete Transient data by key
     * 
     * @param type $key
     * @param type $content
     * @param type $folder
     */
    protected static function _delete_transient_by_key($key, $folder) {
        global $wpdb;
        
        $key_name = 'nasa_cache_' . md5($key) . self::$_subkey . md5($folder);
    
        return $wpdb->query('DELETE FROM ' . $wpdb->options . ' WHERE `option_name` LIKE "_transient_' . $key_name . '%" OR `option_name` LIKE "_transient_timeout_' . $key_name . '%"');
    }
    
    /**
     * Delete File data by key
     * 
     * @param type $key
     * @param type $content
     * @param type $folder
     */
    protected static function _delete_file_by_key($key, $folder) {
        global $nasa_cache_dir;

        if (!isset($nasa_cache_dir) || !$nasa_cache_dir) {
            $upload_dir = wp_upload_dir();
            $nasa_cache_dir = $upload_dir['basedir'] . '/nasa-cache';

            $GLOBALS['nasa_cache_dir'] = $nasa_cache_dir;
        }

        $file = $nasa_cache_dir . '/' . $folder . '/' . md5($key) . '.html';
        if (is_file($file)) {
            wp_delete_file($file);
            
            return true;
        }

        return false;
    }

    /**
     * Delete all cache in any folder
     * 
     * @param type $folder
     */
    public static function delete_cache($folder = 'nasa-core') {
        global $nasa_opt;
        
        /**
         * Cache Mode
         */
        $cache_mode = isset($nasa_opt['enable_nasa_cache']) && $nasa_opt['nasa_cache_mode'] == 'transient' ? 'transient' : 'file';
        
        /**
         * Set Transient
         */
        if ($cache_mode == 'transient') {
            return self::_delete_transients($folder);
        }
        
        /**
         * Set File
         */
        return self::_delete_files($folder);
    }
    
    /**
     * Delete Transient data by key
     * 
     * @param type $key
     * @param type $content
     * @param type $folder
     */
    protected static function _delete_transients($folder) {
        global $wpdb;
        
        $key_name = 'nasa_cache_%' . self::$_subkey . md5($folder);
    
        return $wpdb->query('DELETE FROM ' . $wpdb->options . ' WHERE `option_name` LIKE "_transient_' . $key_name . '%" OR `option_name` LIKE "_transient_timeout_' . $key_name . '%"');
    }
    
    /**
     * Delete File data by key
     * 
     * @param type $key
     * @param type $content
     * @param type $folder
     */
    protected static function _delete_files($folder) {
        global $wp_filesystem, $nasa_cache_dir;

        if (!isset($nasa_cache_dir) || !$nasa_cache_dir) {
            $upload_dir = wp_upload_dir();
            $nasa_cache_dir = $upload_dir['basedir'] . '/nasa-cache';

            $GLOBALS['nasa_cache_dir'] = $nasa_cache_dir;
        }

        // Initialize the WP filesystem, no more using 'file-put-contents' function
        if (empty($wp_filesystem)) {
            require_once ABSPATH . '/wp-admin/includes/file.php';
            WP_Filesystem();
        }

        $folder_cache = $nasa_cache_dir . '/' . $folder;
        if (is_dir($folder_cache)) {
            return $wp_filesystem->rmdir($folder_cache, true);
        }

        return false;
    }
}
